% function I = f_undistort_image(Idistorted, params)
% 
% DESCRIPTION: Undistorts a dense, 3-channel image using the camera parameters.
% 
% PARAMETERS:
%   (INPUT)  Idistorted is a uint8 image
%   (INPUT)  params is a struct composed of alpha,beta,gamma,u0,v0,k1,k2,k3,p1,p2
%   (OUTPUT) uint8 image I
% 
% SAMPLE USAGE:
%   Idistorted = imread('distorted.png');
%   params = struct('alpha',A(1,1),'beta',A(2,2),'gamma',A(1,2),'u0',A(1,3),'v0',A(2,3),...
%                   'k1',k1,'k2',k2,'k3',k3,'p1',p1,'p2',p2);
%   I = f_undistort_image(Idistorted, params);
%   subplot(121); imagesc(Idistorted);
%   subplot(122); imagesc(I);
%   
% SOURCE:
%   Adapted by Klaus Strobl from http://stackoverflow.com/questions/12117825/how-can-i-undistort-an-image-in-matlab-using-the-known-camera-parameters
%   Changes:
%     - Now usable w/o the image processing toolbox by explicitly separating the channels and unit8 casting
%     - The decentering parameters p1,p2 are permuted (historical mistake in Bouguet's
%       toolbox and in OpenCV -- not in DLR CalLab, refer to Weng's 1992 publication)
%     - Bug removed by using the temporal x_ vector
% 
function I = f_undistort_image(Idistorted, params)
% computer vision - How can I undistort an image in Matlab using the known camera parameters? - Stack Overflow
% 
fx = params.alpha;
fy = params.beta;
cx = params.u0;
cy = params.v0;
k1 = params.k1;
k2 = params.k2;
k3 = params.k3;
p1 = params.p1;
p2 = params.p2;

K = [fx 0 cx; 0 fy cy; 0 0 1];

I = zeros(size(Idistorted),'uint8');

%[i j] = find(~isnan(I)); % with image processing toolbox
 [i j] = find(~isnan(I(:,:,1)));

% Xp = the xyz vals of points on the z==1 plane corresponding to undistorted image projections
Xp = inv(K)*[j i ones(length(i),1)]';

% Now we calculate how those points distort i.e forward map them through the distortion
r2 = Xp(1,:).^2+Xp(2,:).^2;
x = Xp(1,:);
y = Xp(2,:);

x_ = x.*(1+k1*r2 + k2*r2.^2) + 2*p2.*x.*y + p1*(r2 + 2*x.^2);
y  = y.*(1+k1*r2 + k2*r2.^2) + 2*p1.*x.*y + p2*(r2 + 2*y.^2);
x  = x_;

% u and v are now the actual, distorted projection positions of ideally undistorted projections
%u = reshape(fx*x + cx,size(I)); % with image processing toolbox
%v = reshape(fy*y + cy,size(I)); % with image processing toolbox
u = reshape(fx*x + cx,size(I(:,:,1)));
v = reshape(fy*y + cy,size(I(:,:,1)));

% Now we perform a backward mapping in order to undistort the warped image coordinates
I(:,:,1) = interp2(single(Idistorted(:,:,1)), u, v);
I(:,:,2) = interp2(single(Idistorted(:,:,2)), u, v);
I(:,:,3) = interp2(single(Idistorted(:,:,3)), u, v);
%I = interp2(Idistorted, u, v); % with image processing toolbox

